﻿
using Hims.Api.Utilities;
using System;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Authorization;
using Hims.Domain.Services;
using Hims.Infrastructure.Services;
using Hims.Shared.EntityModels;
using Hims.Shared.UserModels;
using Hims.Domain.Entities;
using Hims.Api.Models;
using Hims.Shared.DataFilters;
using Hims.Shared.Library.Enums;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{ /// <inheritdoc />
    [Authorize]
    [Route("api/doctor-unit")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DoctorUnitController : BaseController
    {

        /// <summary>
        /// The dcotor unit services.
        /// </summary>
        private readonly IDoctorUnitService doctorUnitServices;
        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public DoctorUnitController(IDoctorUnitService doctorUnitServices, IAuditLogService auditLogService)
        {
            this.doctorUnitServices = doctorUnitServices;
            this.auditLogServices = auditLogService;

        }


        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] DoctorUnitMasterModel model, [FromHeader] LocationHeader header)
        {
            model = (DoctorUnitMasterModel)EmptyFilter.Handler(model);
            var response = await this.doctorUnitServices.InsertAsync(model);
                switch (response )
                {
                case -1:
                    return this.Conflict("Given Unit Name already existed");
                case 0:
                    return this.ServerError();
                default:
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.DoctorUnit,
                        LogDate = DateTime.Now,
                        LogFrom = (short)model.LoginRoleId,
                        LogDescription = $"{model.CreatedByName} has added <b>Doctor Unit Name</b> of <strong>{model.UnitName}</strong> successfully.",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                    return this.Success("UnitName has been added Successfull");
            }
           
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] DoctorUnitMasterModel model)
        {
            model = (DoctorUnitMasterModel)EmptyFilter.Handler(model);
            var response = await this.doctorUnitServices.FetchAllAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// modify status asynchronous.
        /// </summary>
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DoctorUnitMasterModel model, [FromHeader] LocationHeader header)
        {
            model = (DoctorUnitMasterModel)EmptyFilter.Handler(model);
            var response = await this.doctorUnitServices.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.DoctorUnit,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.ModifiedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} Doctor Unit  of  {model.UnitName}  successfully",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }
        /// <summary>
        /// update asynchronous.
        /// </summary>

        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] DoctorUnitMasterModel model, [FromHeader] LocationHeader header)
        {
            model = (DoctorUnitMasterModel)EmptyFilter.Handler(model);
            var response = await this.doctorUnitServices.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Unit Name already existed");
                case 0:
                    return this.ServerError();
                default:
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.DoctorUnit,
                        LogDate = DateTime.Now,
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),
                        LogDescription = $"{model.CreatedByName} has updated <b>Doctor Unit Name</b> of <strong>{model.UnitName}</strong> successfully.",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                    return this.Success("UnitName update successfully");

            }
        }
        
    }
}
